/**************************************************************************************

Copyright (c) Hilscher Gesellschaft fuer Systemautomation mbH. All Rights Reserved.

***************************************************************************************

  $Id: Main.c $:

  Description:
    CIFx driver API definition file

  Changes:
    Date        Description
    -----------------------------------------------------------------------------------
    2013-02-23  initial version

**************************************************************************************/

#include <Windows.h>
#include <stdio.h>
#include <conio.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "netXTransport.h"
#include "cifXErrors.h"
#include "cifXUser.h"
#include "rcX_Public.h"
#include "TL_Marshaller.h"
#include "TL_rcXPacket.h"
#include "netxapi.h"

/* global definitions */
#define CIFX_DEV_NO      0              /* default device number                                */
#define FIRMWARE_PATH    "./"           /* default path to firmware (see TryFirmwareDownload()) */
#define FIRMWARE_NAME    "CIFXECS.NXF"  /* default name of firmware (see TryFirmwareDownload()) */

/* cifX demo application specific function declaration */
void RuncifXDemo(void);

/* global variables */
DWORD  g_dwThreadID       = 0;     /* ID of the cyclic thread (netXTransport cyclic thread)     */
HANDLE g_hThread          = NULL;  /* Handle of the cyclic thread (netXTransport cyclic thread) */
int    g_fRunCyclicThread = 0;     /* state of cyclic thread                                    */

/* function declaration */
/* netXTransport specific functions */
int32_t RS232_Connector_Init  (void);
void    RS232_Connector_Deinit(void);
int32_t TCP_Connector_Init    (void);
void    TCP_Connector_Deinit  (void);

/*****************************************************************************/
/*****************************************************************************/
/*! FUNCTIONS REQUIRED FOR NETXTRANSPORT INITIALIZATION                      */
/*****************************************************************************/
/*****************************************************************************/

/*****************************************************************************/
/*! Initializes the cifX Marshaller Translation-Layer and the rcX-Packet
*   Translation-Layer.
*   \param pvParam    User param (currently not used)
*   \return NXT_NO_ERROR on success                                          */
/*****************************************************************************/
int32_t APIENTRY TLLayerInit( void* pvParam)
{
  int32_t lRet = NXT_NO_ERROR;

  if (NXT_NO_ERROR == (lRet = cifX_Marshaller_Init( pvParam)))
  {
    lRet = rcXPacket_Init( pvParam);
  }
  return lRet;
}

/*****************************************************************************/
/*! De-initializes translation layer
*   \param pvParam    User param (currently not used)                        */
/*****************************************************************************/
void APIENTRY TLLayerDeInit( void* pvParam)
{
  cifX_Marshaller_DeInit( pvParam);
  rcXPacket_DeInit( pvParam);
}

/*****************************************************************************/
/*! Starts the cyclic thread of the netXTransport Toolkit
*   (required for periodic jobs).
*   \param lpParameter  User param (currently not used)
*   \return 0 on success                                                     */
/*****************************************************************************/
DWORD WINAPI CyclicThread( LPVOID lpParameter)
{
  UNREFERENCED_PARAMETER( lpParameter);

  g_fRunCyclicThread = 1;

  while( g_fRunCyclicThread == 1)
  {
    Sleep(100);

    netXTransportCyclicFunction();
  }
  return 0;
}


/*****************************************************************************/
/*! Demo-Application

     The demo application initializes the netXTransport toolkit, supporting the
     Translation-Layer rcX-Packet (Data-Type=0x100) and cifXMarshaller
     (Data-Type=0x200).
     The application provides two connector types (RS232/USB and TCP) which can
     be selected during runtime.
     If a netX device is successfully detected, the cifX demo application starts.

     To be able to use the cifX API interface, 4 steps are required (see code below):
      1. Initialize netXTransport Toolkit (choose appropriate Translation-Layer)
      2. Add a connector (e.g. RS232/USB, TCP)
      3. Start the netXTransport Toolkit (starts device discovering process)
      4. Start the netXTransport Cyclic Thread (enable cyclic jobs)
                                                                             */
/*****************************************************************************/
int main(void)
{
  TL_INIT_T                tDataLayerInit;
  int32_t                  lRet        = NXT_NO_ERROR;

  /* setup netXTransport initialization structure */
  tDataLayerInit.pfnTLInit   = TLLayerInit;   /* function pointer the Translation-Layer initialization    */
  tDataLayerInit.pfnTLDeInit = TLLayerDeInit; /* function pointer the Translation-Layer de-initialization */
  tDataLayerInit.pvData      = NULL;          /* Private data (currently  not used)                       */

  printf("\n**************************************************************\n");
  printf("\nStarting netXTransport Demo Application...\n");
  printf("\nThe netXTransport demonstration, using the example of the cifX-API.\n");
  printf("\n**************************************************************\n");
  printf("\nInitialize the netXTransport Toolkit...\n");

  /*****************************************************************************/
  /* 1. Initialize netXTransport Toolkit (add Translation-Layer)               */
  /*****************************************************************************/
  if (NXT_NO_ERROR != (lRet = netXTransportInit( &tDataLayerInit, sizeof(tDataLayerInit))))
  {
    printf("\nError during toolkit initialization. lRet=0x%X\n", lRet);
  }

  /*****************************************************************************/
  /* 2. Add a connector                                                        */
  /*****************************************************************************/
  if (lRet == NXT_NO_ERROR)
  {
    int iInp;
    printf("\n**************************************************************\n");
    printf("Enable connector ('r'=RS232 / 't'=TCP / 'b'=both):");
    iInp = toupper(getchar());

    if ((iInp == 'R') || (iInp == 'B'))
    {
      printf("\nAdd example connector (RS232/USB)...\n");
      if (NXT_NO_ERROR != (lRet = RS232_Connector_Init()))
      {
        printf("\nError during RS232-Connector initialization. lRet=0x%X\n", lRet);
      }
    }
    if ((iInp == 'T') || (iInp == 'B'))
    {
      printf("\nAdd example connector TCP...\n");
      if (NXT_NO_ERROR != (lRet = TCP_Connector_Init()))
      {
        printf("\nError during TCP-Connector initialization. lRet=0x%X\n", lRet);
      }
    }
  }

  /*****************************************************************************/
  /* 3. Start the netXTransport Toolkit                                        */
  /*****************************************************************************/
  if (lRet == NXT_NO_ERROR)
  {
    printf("\nStart netXTransport Toolkit...\n");

    /* start netXTransport (starts device discovering and registration) */
    if (NXT_NO_ERROR != (lRet = netXTransportStart( NULL, NULL)))
    {
      printf("\nError while trying to start. lRet=0x%X\n", lRet);
    }
  }
  if (lRet != NXT_NO_ERROR)
  {
    printf("\nAborting demo application!\n");
    /* de-initialize toolkit */
    netXTransportDeinit();
    RS232_Connector_Deinit();
    TCP_Connector_Deinit();

    system("Pause");
    return 0;
  }

  /*****************************************************************************/
  /* 4. Start the netXTransport Cyclic Thread (required for asynchronous cyclic
        jobs, like Keep-Alive).                                                */
  /*****************************************************************************/
  g_hThread = CreateThread( NULL, 0, CyclicThread, NULL, 0, &g_dwThreadID);

  /*****************************************************************************/
  /* 5. Start the Application                                                  */
  /*****************************************************************************/
  /* NOTE: Now its possible to use the standard cifX API functions, depending on
  *        the registerd data layer in this case we can use the cifXAPI
  *        (cifXAPI -> supports packet types (data layer/packet type) rcXPackets
  *        (0x100) and cifXMarshaller (0x200).
  *        For more information of cifXAPI usage refer to the cifXAPI
  *        documentation ().                                                   */
  RuncifXDemo();

  /*****************************************************************************/
  /* 5. Deinitialization process                                               */
  /*****************************************************************************/
  printf("De-initialize the netXTransport Toolkit...\n");

  /* stop cyclic thread */
  g_fRunCyclicThread = 0;

  if (g_hThread != NULL)
  {
    if (WAIT_TIMEOUT == WaitForSingleObject( g_hThread, 1000))
      TerminateThread( g_hThread, 0xFFFFFFFF);
  }

  netXTransportStop();
  RS232_Connector_Deinit();
  TCP_Connector_Deinit();
  netXTransportDeinit();

  printf("\n\nnetXTransport Demo ended!\n");
  system("Pause");

  return 0;
}

/*****************************************************************************/
/*****************************************************************************/
/*! FUNCTIONS RELATED TO THE CIFX API DEMO USAGE                             */
/*****************************************************************************/
/*****************************************************************************/

/*****************************************************************************/
/*! Copies firmware file into ppbBuffer
*   \param szFileName       Path to firmware file
*   \param ppbBuffer        Pointer that receives pointer to buffer
*   \param pulFileSize      Size of ppbBuffer
*   \return CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t GetFirmware( char* szFileName, uint8_t** ppbBuffer, uint32_t* pulFileSize)
{
  int32_t      lRet       = CIFX_FUNCTION_FAILED;
  FILE*        hFile      = NULL;
  struct _stat tStat;
  size_t       ret;
  int          iRet;

  if (0 == _stat( szFileName, &tStat))
  {
    lRet       = CIFX_NO_ERROR;
    *ppbBuffer = (uint8_t*)malloc( tStat.st_size);

    if (0 == (iRet= fopen_s( &hFile, szFileName, "rb")))
    {
      if (0!=(ret = fread(*ppbBuffer, 1, tStat.st_size, hFile)))
      {
        *pulFileSize = tStat.st_size;
        return CIFX_NO_ERROR;
      }
    }
  }
  return lRet;
}

/*****************************************************************************/
/*! Checks if file exists on device
*   \param hChannel    Handle to an opened channel
*   \param szFileName  Name of the file
*   \return 1 if it exists                                                   */
/*****************************************************************************/
int CheckIfFileExists(CIFXHANDLE hChannel, char* szFileName)
{
  int32_t             lRet           = CIFX_NO_ERROR;
  int                 fFound         = 0;
  CIFX_DIRECTORYENTRY tDirectoryInfo = {0};

  if (CIFX_NO_ERROR == (lRet = xSysdeviceFindFirstFile(  hChannel, 0, &tDirectoryInfo, NULL, NULL)))
  {
    while(lRet == CIFX_NO_ERROR)
    {
      lRet = xSysdeviceFindNextFile( hChannel, 0, &tDirectoryInfo, NULL, NULL);

      if (strcmp(tDirectoryInfo.szFilename, szFileName) == 0)
      {
        fFound = 1;
        break;
      }
    }
  }
  return fFound;
}

/*****************************************************************************/
/*! Shows the error description
*   \param lError  Error value                                               */
/*****************************************************************************/
void ShowError( int32_t lError)
{
  if( lError != CIFX_NO_ERROR)
  {
    /* Read driver error description */
    char szError[1024] ={0};
    xDriverGetErrorDescription( lError,  szError, sizeof(szError));
    printf("Error: 0x%X, <%s>\r\n", lError, szError);
  }
}

/*****************************************************************************/
/*! Writes channel information to console
*   \param hDriver     Handle to the opened driver
*   \param ulBoardIdx  Number of the board                                   */
/*****************************************************************************/
void ShowChannelInfo( CIFXHANDLE hDriver, uint32_t ulBoardIdx)
{
  unsigned long ulChannelIdx  = 0;
  int32_t       lChannelRet   = CIFX_NO_ERROR;
  int           fBoardFound   = 0;

  while(lChannelRet == CIFX_NO_ERROR)
  {
    /* Read all channel information from the given board */
    CHANNEL_INFORMATION tChannelInfo = {{0}};
    lChannelRet = xDriverEnumChannels(hDriver, ulBoardIdx, ulChannelIdx, sizeof(tChannelInfo), &tChannelInfo);
    if(lChannelRet != CIFX_NO_ERROR)
    {
      if (lChannelRet != CIFX_NO_MORE_ENTRIES)
      {
        /* Show information */
        printf("Error during xDriverEnumChannels(): Channel %u\r\n", (unsigned int)ulChannelIdx);
        ShowError(lChannelRet);
      }
    } else
    {
      /* We have a board and a channel */
      fBoardFound = 1;

      /* Show information */
      printf("  Channel%u Information:\r\n", (unsigned int)ulChannelIdx);
      printf("   Channel Error            : 0x%08X\r\n",  tChannelInfo.ulChannelError);
      printf("   Board Name               : %s\r\n",  tChannelInfo.abBoardName);
      printf("   Alias Name               : %s\r\n",  tChannelInfo.abBoardAlias);
      printf("   Device Nr.               : %lu\r\n",  (long unsigned int)tChannelInfo.ulDeviceNumber);
      printf("   Serial Nr.               : %lu\r\n",  (long unsigned int)tChannelInfo.ulSerialNumber);
      printf("   MBX Size                 : %lu\r\n",  (long unsigned int)tChannelInfo.ulMailboxSize);
      printf("   Firmware Name            : %s\r\n",  tChannelInfo.abFWName);
      printf("   Firmware Version         : %d.%d.%d Build %d\r\n",  tChannelInfo.usFWMajor, tChannelInfo.usFWMinor,tChannelInfo.usFWRevision, tChannelInfo.usFWBuild);
      printf("   Open Counter             : %lu\r\n",  (long unsigned int)tChannelInfo.ulOpenCnt);
      printf("   Put Packet Counter       : %lu\r\n",  (long unsigned int)tChannelInfo.ulPutPacketCnt);
      printf("   Get Packet Counter       : %lu\r\n",  (long unsigned int)tChannelInfo.ulGetPacketCnt);
      printf("   Number of IO Input Areas : %lu\r\n",  (long unsigned int)tChannelInfo.ulIOInAreaCnt);
      printf("   Number of IO Output Areas: %lu\r\n",  (long unsigned int)tChannelInfo.ulIOOutAreaCnt);
      printf("   Size of handshake cells  : %lu\r\n",  (long unsigned int)tChannelInfo.ulHskSize);
      printf("   Actual netX Flags        : 0x%08X\r\n",tChannelInfo.ulNetxFlags);
      printf("   Actual host Flags        : 0x%08X\r\n",tChannelInfo.ulHostFlags);
    }
    ++ulChannelIdx;    /* Next channel */
  }
}

/*****************************************************************************/
/*! Packet handling demo
*   \param hDriver      Handle to the opened driver
*   \param szBoardName  Name of the board                                    */
/*****************************************************************************/
void PacketDemo(CIFXHANDLE hDriver, char* szBoardName)
{
  CIFXHANDLE  hSysdevice  = NULL;
  CIFX_PACKET tSendPacket = {0};
  CIFX_PACKET tRecvPacket = {0};
  int32_t     lRet        = CIFX_NO_ERROR;
  uint32_t    ulVisible   = 5;
  uint32_t    ulLoopCnt   = 0;

  printf("\nRunning test on system device...\n");

  if(CIFX_NO_ERROR == (xSysdeviceOpen(hDriver, szBoardName, &hSysdevice)))
  {
    NETX_SYSTEM_STATUS_BLOCK tSystemSatusBlock;

    if (CIFX_NO_ERROR != (lRet = xSysdeviceInfo(hSysdevice,
                           CIFX_INFO_CMD_SYSTEM_STATUS_BLOCK,
                           sizeof(tSystemSatusBlock),
                           &tSystemSatusBlock)))
    {
      printf("Failed xSysdeviceInfo() request (lRet = 0x%X)\n", lRet);

    }
    memset(&tSendPacket.tHeader, 0, sizeof(tSendPacket.tHeader));
    memset(&tRecvPacket.tHeader, 0, sizeof(tRecvPacket.tHeader));

    tSendPacket.tHeader.ulCmd = RCX_HW_IDENTIFY_REQ;

    printf("\n**************************************************************\n");
    printf("Starting demo - Packet-Communication...\r\n");
    printf("**************************************************************\n");
    printf("Doing sequently xSysdevicePutPacket() - xSysdeviceGetPacket().\n");
    printf("\nTo schedule a reset, press 'r'...\n");
    printf("To interrupt demo press any key...\n\n");
    printf("Packet configured for hw-identify (RCX_HW_IDENTIFY_REQ)\n");
    printf("xSysdevicePutPacket(): ->\n");
    printf("xSysdeviceGetPacket(): <-\n\n");

    do
    {
      /* send packet */
      if (CIFX_NO_ERROR != (lRet = xSysdevicePutPacket(hSysdevice, &tSendPacket, CIFX_TO_SEND_PACKET)))
      {
        printf("xSysdevicePutPacket() failed for packet Paket-ID=%d (lRet=0x%X)\n", tSendPacket.tHeader.ulId, lRet);
        Sleep(500);
      } else
      {
        /* do not print every packet */
        if (ulLoopCnt < ulVisible)
        {
          printf("-> ulCmd=0x%08X, ulId=%d, lRet=0x%08X\n",
              tSendPacket.tHeader.ulCmd, tSendPacket.tHeader.ulId, lRet);
        }

        Sleep(1);
        /* get packet */
        if (CIFX_NO_ERROR != (lRet = xSysdeviceGetPacket(hSysdevice, sizeof(tRecvPacket), &tRecvPacket, CIFX_TO_SEND_PACKET)))
        {
          printf("xSysdeviceGetPacket() failed for packet Paket-ID=%d (lRet=0x%X)\n", tSendPacket.tHeader.ulId, lRet);
        } else
        {
          /* do not print every packet */
          if (ulLoopCnt < ulVisible)
          {
            printf("<- ulCmd=0x%08X, ulId=%d, ulSta=0x%08X, ulLen=%d, lRet=0x%08X\n",
               tRecvPacket.tHeader.ulCmd, tRecvPacket.tHeader.ulId, tRecvPacket.tHeader.ulState, tRecvPacket.tHeader.ulLen, lRet);
          }
        }
      }
      if (ulLoopCnt == ulVisible)
        printf("\nSkip output (except errors and reset instruction)...\n");
      ++tSendPacket.tHeader.ulId;

      if (_kbhit())
      {
        int iInp = toupper(_getch());
        if (iInp == 'R')
        {
          printf("Resetting device...\n");
          if (CIFX_NO_ERROR != (lRet = xSysdeviceReset( hSysdevice, 15000)))
          {
            printf("Error during reset!!! (lRet = 0x%X)\n", lRet);
            printf("Demo canceled...\n");
            break;
          }
          ulLoopCnt = 1;
        } else
        {
          printf("Continue packet demo...\n");
          printf("Demo canceled by user...\n");
          break;
        }
      }
      ulLoopCnt++;
    } while(1);
    printf("\n");
    lRet = xSysdeviceClose(hSysdevice);

  } else
  {
    printf("Error open system device (lRet=0x%X)\n", lRet);
  }
}

/*****************************************************************************/
/*! Writes channel information to console
*   \param hDriver      Handle to the opened driver
*   \param szBoardName  Name of the board                                    */
/*****************************************************************************/
void TryFirmwareDownload( CIFXHANDLE hDriver, char* szBoardName)
{
  int32_t             lRet         = CIFX_NO_ERROR;
  CIFXHANDLE          hChannel     = NULL;
  uint8_t*            pabFileData  = NULL;
  uint32_t            ulFileSize   = 0;

  printf("\nTry to open channel 0 of device \"%s\"\n", szBoardName);
  if (CIFX_NO_ERROR == (lRet = xSysdeviceOpen( hDriver, szBoardName, &hChannel)))
  {
    if (CheckIfFileExists( hChannel, FIRMWARE_NAME))
    {
      printf("\nSkip firmware download. Firmware already existing!\n");
    } else
    {
      char* strFirmware = NULL;

      if (NULL != (strFirmware = malloc(strlen(FIRMWARE_PATH)+strlen(FIRMWARE_NAME)+1)))
      {
        sprintf_s( strFirmware, (strlen(FIRMWARE_PATH)+strlen(FIRMWARE_NAME)+1), "%s%s", FIRMWARE_PATH, FIRMWARE_NAME);

        if (CIFX_NO_ERROR == GetFirmware( strFirmware, &pabFileData, &ulFileSize))
        {
          printf("\nTry to download the firmware %s\n", strFirmware);
          /* try firmware download */
          if (CIFX_NO_ERROR == (lRet = xSysdeviceDownload( hChannel,
                                     0,
                                     DOWNLOAD_MODE_FIRMWARE,
                                     FIRMWARE_NAME,
                                     pabFileData,
                                     ulFileSize,
                                     NULL,
                                     NULL,
                                     NULL)))
          {
            if (CheckIfFileExists( hChannel, FIRMWARE_NAME))
              printf("Successfully downloaded the firmware\n");
            else
              printf("Can not verify download result!\n");
          } else
          {
            printf("Error while downloading the firmware (lRet=0x%X)\n", lRet);
          }
        } else
        {
          printf("Skip firmware download. Error while retrieving the firmware\n");
        }
      } else
      {
        printf("Error while allocating memory for file download! Skip firmware download\n");
      }
    }
    lRet = xSysdeviceClose(hChannel);

  } else
  {
    printf("Error opening channel 0 (lRet=0x%X)\n", lRet);
  }
  if (pabFileData)
    free(pabFileData);
}

/*****************************************************************************/
/*! Enumerates a board given by the number.
*   \param hDriver      Handle to the opened driver
*   \param BoardNo      Handle to the opened driver
*   \param szBoardName  return the name of the board
*   return  CIFX_NO_ERROR on success                                         */
/*****************************************************************************/
int32_t EnumBoards( CIFXHANDLE hDriver, uint32_t BoardNo, char* szBoardName)
{
  int32_t           lRet       = CIFX_NO_ERROR;
  BOARD_INFORMATION tBoardInfo = {0};

  /* discover boards */
  if (CIFX_NO_ERROR == (lRet = xDriverEnumBoards( hDriver, BoardNo, sizeof(tBoardInfo), &tBoardInfo)))
  {
    printf("Board%u Information:\r\n", (unsigned int)BoardNo);
    printf(" Name : %s\r\n", tBoardInfo.abBoardName);
    printf(" Alias: %s\r\n", tBoardInfo.abBoardAlias);
    printf(" DevNr: %lu\r\n", (long unsigned int)tBoardInfo.tSystemInfo.ulDeviceNumber);
    printf(" SN   : %lu\r\n", (long unsigned int)tBoardInfo.tSystemInfo.ulSerialNumber);
    printf("\r\n");

    strcpy_s( szBoardName, sizeof(tBoardInfo.abBoardName), tBoardInfo.abBoardName);
  } else
  {
    printf("Error while enumerating available devices (lRet=0x%X)\n", lRet);
  }
  return lRet;
}

/*****************************************************************************/
/*! Starts cifX demo application                                             */
/*****************************************************************************/
void RuncifXDemo(void)
{
  CIFXHANDLE hDriver = NULL;
  int32_t    lRet    = CIFX_NO_ERROR;
  uint32_t   BoardNo = CIFX_DEV_NO;
  char       szBoardName[256];

  /* open the driver */
  if (CIFX_NO_ERROR == xDriverOpen(&hDriver))
  {
    DRIVER_INFORMATION tDriverInfo;
    if (CIFX_NO_ERROR != (lRet = xDriverGetInformation( hDriver, sizeof(tDriverInfo), &tDriverInfo)))
    {
      printf("Error while requesting the driver information (lRet=0x%X)!\n", lRet);
    } else
    {
      printf("Driver Version: %s\n", tDriverInfo.abDriverVersion);
      printf("Device Count  : %d\n", tDriverInfo.ulBoardCnt);

      if (tDriverInfo.ulBoardCnt)
      {
        printf("Select board number:");
        scanf_s("%d",&BoardNo);

      } else
      {
        /* force board 0 */
         BoardNo = 0;
      }
    }
    if (CIFX_NO_ERROR == EnumBoards( hDriver, BoardNo, szBoardName))
    {
      ShowChannelInfo( hDriver, BoardNo);

      TryFirmwareDownload( hDriver, szBoardName);

      PacketDemo( hDriver, szBoardName);
    }
    lRet = xDriverClose( hDriver);
  }
}